<?php

namespace Opencart\Catalog\Controller\Extension\SebPaymentGateway\Payment;

use Opencart\System\Engine\Config;
use Opencart\System\Engine\Controller;
use Opencart\System\Library\SebPaymentGateway as SebPaymentGatewayLibrary;

$GLOBALS['sebpaymentgateway'] = array(
	'urlLive' => 'https://payment.ecommerce.sebgroup.com/api/v3/payments/oneoff',
	'urlTest' => 'https://igw-demo.every-pay.com/api/v3/payments/oneoff',
	'urlLiveStatus' => 'https://payment.ecommerce.sebgroup.com/api/v3/payments/',
	'urlTestStatus' => 'https://igw-demo.every-pay.com/api/v3/payments/',
);

class SebPaymentGateway extends Controller {
	protected $globalURLs;
	private $separator = '';

	public function __construct($registry) {
		parent::__construct($registry);

		header('Set-Cookie: ' . $this->config->get('session_name') . '=' . $this->session->getId() . '; SameSite=Lax');

		if (VERSION >= '4.0.2.0') {
			$this->separator = '.';
		} else {
			$this->separator = '|';
		}
	}

	public function index() {
		$this->load->language('extension/sebpaymentgateway/payment/sebpaymentgateway');

		$data['button_payment_confirmation'] = ($this->language->get('button_payment_confirmation')
			&& (mb_strlen($this->language->get('button_payment_confirmation')) > 0))
			? $this->language->get('button_payment_confirmation')
			: 'Proceed to Payment';
		$data['payment_url'] = $this->url->link('extension/sebpaymentgateway/payment/sebpaymentgateway' . $this->separator . 'redirect', true);

		return $this->load->view('extension/sebpaymentgateway/payment/sebpaymentgateway', $data);
	}

	public function redirect() {
		include_once DIR_EXTENSION . 'sebpaymentgateway/system/library/sebpaymentgateway.php';
		$this->load->language('extension/sebpaymentgateway/payment/sebpaymentgateway');
		$this->load->model('checkout/order');
		$this->load->model('extension/sebpaymentgateway/payment/sebpaymentgateway');

		$order = $this->model_checkout_order->getOrder($this->session->data['order_id']);
		$storeName = $this->config->get('config_name');

		$countryCode = $this->getBillingItem($order, 'iso_code_2');

		$fields = array(
			'account_name' => $this->config->get('payment_sebpaymentgateway_account_name'),
			'amount' => $order['total'],
			'billing_city' => $this->getBillingItem($order, 'city'),
			'billing_country' => $countryCode,
			'billing_line1' => $this->getBillingItem($order, 'address_1'),
			'billing_line2' => $this->getBillingItem($order, 'address_2'),
			'billing_postcode' => $this->getBillingItem($order, 'postcode'),
			'customer_url' => $this->url->link('extension/sebpaymentgateway/payment/sebpaymentgateway' . $this->separator . 'customerReturn', true),
			'shipping_city' => $order['shipping_city'],
			'shipping_country' => $order['shipping_iso_code_2'],
			'shipping_line1' => $order['shipping_address_1'],
			'shipping_line2' => $order['shipping_address_2'] ?: '-',
			'shipping_code' => $order['shipping_postcode'],
			'shipping_state' => $order['shipping_zone'],
			'email' => $order['email'],
			'order_reference' => $order['order_id'],
			'customer_ip' => $order['ip'],
			'integration_details' => [
				'integration' => 'SEB Plugin',
				'version' => '1.0.0',
				'software' => 'OpenCart ' . VERSION,
			],
		);

		if (!empty($this->getPreferredCountry($countryCode))) {
			$fields['preferred_country'] = $countryCode;
		}

		$sebpaymentgateway = new SebPaymentGatewayLibrary();
		$config = $this->getConfig();

		if ($config['mode'] === 'live') {
			$fields['api_username'] = $this->config->get('payment_sebpaymentgateway_api_username');
			$sebpaymentgateway->init($this->config->get('payment_sebpaymentgateway_api_username'), $this->config->get('payment_sebpaymentgateway_api_secret'));
		} else {
			$fields['api_username'] = $this->config->get('payment_sebpaymentgateway_test_api_username');
			$sebpaymentgateway->init($this->config->get('payment_sebpaymentgateway_test_api_username'), $this->config->get('payment_sebpaymentgateway_test_api_secret'));
		}

		$data['form_data'] = $sebpaymentgateway->getFields($fields, $order['language_code']);

		$this->globalURLs =& $GLOBALS['sebpaymentgateway'];

		if ($config['mode'] === 'live') {
			$username = $this->config->get('payment_sebpaymentgateway_api_username');
			$password = $this->config->get('payment_sebpaymentgateway_api_secret');
			$url = $this->globalURLs['urlLive'];
		} else {
			$username = $this->config->get('payment_sebpaymentgateway_test_api_username');
			$password = $this->config->get('payment_sebpaymentgateway_test_api_secret');
			$url = $this->globalURLs['urlTest'];
		}

		$request = $this->sendPostRequestWithBasicAuth($url, $data['form_data'], $username, $password);
		$result = json_decode($request, true);

		return $this->response->redirect($result['payment_link'] ?? '/');
	}

	private function getPreferredCountry($countryCode) {
		$allowedCountryCodes = ['EE', 'LV', 'LT'];

		if (in_array($countryCode, $allowedCountryCodes)) {
			return $countryCode;
		} else {
			return '';
		}
	}

	private function getBillingItem(array $order, string $itemName): string {
		if (!empty($order['payment_' . $itemName])) {
			return $order['payment_' . $itemName];
		} elseif (!empty($order['shipping_' . $itemName])) {
			return $order['shipping_' . $itemName];
		} else {
			return '-';
		}
	}

	private function sendPostRequestWithBasicAuth($url, $data, $username, $password) {
		$request_body = json_encode($data);
		$ch = curl_init($url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLINFO_HEADER_OUT, true);
		curl_setopt($ch, CURLOPT_POST, true);
		curl_setopt($ch, CURLOPT_POSTFIELDS, $request_body);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array(
			'Content-Type: application/json',
			'Accept: application/json',
			'Content-Length: ' . strlen($request_body))
		);
		curl_setopt($ch, CURLOPT_USERPWD, $username . ":" . $password);
		$response = curl_exec($ch);
		curl_close($ch);
		return $response;
	}

	private function sendGetRequestWithAuth($url, $username, $password){
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array(
				'Content-Type: application/json'
			)
		);
		curl_setopt($ch, CURLOPT_USERPWD, $username . ":" . $password);
		$response = curl_exec($ch);
		curl_close($ch);
		return json_decode($response, true);
	}

	private function getPaymentResultUrl($payment_reference, $api_username) {
		include_once DIR_EXTENSION . 'sebpaymentgateway/system/library/sebpaymentgateway.php';
		$globalURLs = $this->globalURLs =& $GLOBALS['sebpaymentgateway'];
		$sebpaymentgateway = new SebPaymentGatewayLibrary();
		$config = $this->getConfig();

		if ($config['mode'] === 'live') {
			$sebpaymentgateway->init($this->config->get('payment_sebpaymentgateway_api_username'), $this->config->get('payment_sebpaymentgateway_api_secret'));
			$url = $globalURLs['urlLiveStatus'];
		} else {
			$sebpaymentgateway->init($this->config->get('payment_sebpaymentgateway_test_api_username'), $this->config->get('payment_sebpaymentgateway_test_api_secret'));
			$url = $globalURLs['urlTestStatus'];
		}

		return $url . $payment_reference . "?api_username=" . $api_username;
	}

	public function process() {
		header('Set-Cookie: ' . $this->config->get('session_name') . '=' . $this->session->getId() . '; SameSite=Lax');

		include_once DIR_EXTENSION . 'sebpaymentgateway/system/library/sebpaymentgateway.php';
		$this->load->model('checkout/order');
		$sebpaymentgateway = new SebPaymentGatewayLibrary();
		$config = $this->getConfig();

		if ($config['mode'] === 'live') {
			$sebpaymentgateway->init($this->config->get('payment_sebpaymentgateway_api_username'), $this->config->get('payment_sebpaymentgateway_api_secret'));
		} else {
			$sebpaymentgateway->init($this->config->get('payment_sebpaymentgateway_test_api_username'), $this->config->get('payment_sebpaymentgateway_test_api_secret'));
		}

		$response = $_REQUEST;
		$payment_reference = $_REQUEST['payment_reference'];
		$order_reference = $_REQUEST['order_reference'];
		$response['api_username'] = $sebpaymentgateway->getApiUsername();
		$response['api_secret'] = $sebpaymentgateway->getApiSecret();

		$stateRequest = $this->sendGetRequestWithAuth($this->getPaymentResultUrl($payment_reference, $response['api_username']), $response['api_username'], $response['api_secret'])["payment_state"];
		$response['payment_state'] = $stateRequest;

		try {
			$this->load->model('extension/sebpaymentgateway/payment/sebpaymentgateway');
			$this->model_extension_sebpaymentgateway_payment_sebpaymentgateway->addOrderPaymentStatus($order_reference, $stateRequest);
		} catch (\Exception $e) {}

		$status = $sebpaymentgateway->verify($response);

		switch($status) {
			case 1:
				$this->model_checkout_order->addHistory($order_reference, (int) $config['order_status_id'], '', true);
				$this->response->redirect($this->url->link('checkout/success'));
				break;
			case 2:
				$this->model_checkout_order->addHistory($order_reference, 7, '', true);
				$this->response->redirect($this->url->link('checkout/checkout'));
				break;
			case 3:
				$this->model_checkout_order->addHistory($order_reference, 10, '', true);
				$this->response->redirect($this->url->link('checkout/failure'));
				break;
			// #19031 here we set order status as 'pending' and show redirect user to 'success' order placement page
			case 4:
				$this->model_checkout_order->addHistory($order_reference, 1, '', true);
				$this->response->redirect($this->url->link('checkout/success'));
				break;
			default:
				$this->response->redirect($this->url->link('checkout/checkout'));
		}
	}

	public function customerReturn() {
		header('Set-Cookie: ' . $this->config->get('session_name') . '=' . $this->session->getId() . '; SameSite=Lax');

		sleep(3);
		$config = $this->getConfig();
		$this->load->model('checkout/order');
		$this->load->model('extension/sebpaymentgateway/payment/sebpaymentgateway');

		$order_reference = htmlentities($_REQUEST['order_reference']);
		if (empty($order_reference)) {
			$this->response->redirect($this->url->link('checkout/checkout'));
		}

		$paymentStatus = $this->model_extension_sebpaymentgateway_payment_sebpaymentgateway->getPaymentStatusByReference($order_reference);
		if (empty($paymentStatus)) {
			$this->displayNoStatusMessage();
			return;
		}

		$order = $this->model_checkout_order->getOrder($order_reference);
		if (empty($order)) {
			$this->response->redirect($this->url->link('checkout/checkout'));
		}

		switch($order['order_status_id']) {
			case (int) $config['order_status_id']:
			case 1:
			case 7:
				$this->response->redirect($this->url->link('checkout/success'));
				break;
			case 10:
				$this->response->redirect($this->url->link('checkout/failure'));
				break;
			default:
				$this->response->redirect($this->url->link('checkout/checkout'));
		}
	}

	private function displayNoStatusMessage() {
		$this->load->language('checkout/checkout');
		$this->load->language('extension/sebpaymentgateway/payment/sebpaymentgateway');

		$data['breadcrumbs'] = array();

		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('text_home'),
			'href' => $this->url->link('common/home')
		);

		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('text_cart'),
			'href' => $this->url->link('checkout/cart')
		);

		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('heading_title'),
			'href' => $this->url->link('checkout/checkout', '', true)
		);

		$data['column_left'] = $this->load->controller('common/column_left');
		$data['column_right'] = $this->load->controller('common/column_right');
		$data['content_top'] = $this->load->controller('common/content_top');
		$data['content_bottom'] = $this->load->controller('common/content_bottom');
		$data['footer'] = $this->load->controller('common/footer');
		$data['header'] = $this->load->controller('common/header');
		$data['message'] = $this->language->get('no_status_message');

		$this->response->setOutput($this->load->view('extension/sebpaymentgateway/payment/sebpaymentgateway_no_status', $data));
	}

	private function getConfig(): array {
		$_config = new Config();
		$_config->addPath(DIR_EXTENSION . 'sebpaymentgateway/system/config/');
		$_config->load('sebpaymentgateway');

		$data = [];
		$data['setting'] = $_config->get('sebpaymentgateway_setting');
		$data['setting'] = array_replace_recursive((array)$data['setting'], (array)$this->config->get('payment_sebpaymentgateway_setting'));

		return $data['setting']['general'];
	}
}
