<?php

namespace Opencart\System\Library;

class SebPaymentGateway {
	const _VERIFY_SUCCESS = 1; // payment successful
	const _VERIFY_CANCEL = 2;  // payment cancelled
	const _VERIFY_FAIL = 3;    // payment failed
	const _VERIFY_PENDING = 4; // payment pending

	private $api_username;
	private $api_secret;
	private $request_cc_token;
	private $cc_token;
	private $statuses = array(
		'settled'                 => self::_VERIFY_SUCCESS,
		'authorised'              => self::_VERIFY_SUCCESS,
		'cancelled'               => self::_VERIFY_CANCEL,
		'waiting_for_3ds_response'=> self::_VERIFY_CANCEL,
		'failed'                  => self::_VERIFY_FAIL,
		'sent_for_processing'     => self::_VERIFY_PENDING
	);

	// allowed locales taken from EveryPay API documentation
	private $allowed_locales = array ('en','et','fi','de','lv','lt','ru','es','sv','da','pl','it','fr','nl','pt','no');

	/**
	* Initiates a payment.
	*
	* Expects following data as input:
	* 'api_username' => api username,
	* 'api_secret => api secret,
	* array(
	*  'request_cc_token' => request to get cc_token 1,
	*  'cc_token' => token referencing a bank card
	* )
	*
	* @param $api_username string
	* @param $api_secret string
	* @param $data array
	* @example $sebpaymentgateway->init('api_username', 'api_secret', array('cc_token' => 'token'));
	*
	* @return void
	*/

	public function init($api_username, $api_secret, $data = '')
	{
		$this->api_username = $api_username;
		$this->api_secret = $api_secret;

		if (isset($data['request_cc_token'])) {
			$this->request_cc_token = $data['request_cc_token'];
		}

		if (isset($data['cc_token'])) {
			$this->cc_token = $data['cc_token'];
		}
	}

	public function getFields(array $data, $language = 'et')
	{
		$data['api_username'] = $this->api_username;
		$data['nonce'] = $this->getNonce();
		$data['timestamp'] = date('c');
		$data['transaction_type'] = 'charge';

		if (isset($this->request_cc_token)) {
			$data['request_cc_token'] = $this->request_cc_token;
		}

		if (isset($this->cc_token)) {
			$data['cc_token'] = $this->cc_token;
		}

		$data['locale'] = $this->getLocale($language);

		return $data;
	}

	public function verify(array $data)
	{
		if ($data['api_username'] !== $this->api_username) {
			throw new \Exception('Invalid username.');
		}

		$status = !empty($data['payment_state']) ? $this->statuses[$data['payment_state']] : false;

		return $status;
	}


	protected function getNonce()
	{
		return uniqid(true);
	}

	protected function verifyNonce($nonce)
	{
		return true;
	}

	public function getApiUsername()
	{
		return $this->api_username;
	}

	public function getApiSecret()
	{
		return $this->api_secret;
	}

	private function getLocale($language)
	{
		$lang_formatted = "en";
		if (mb_strlen($language) >= 2) {
			switch ($language):
				case "eston":
					// this is how language code was derived from language with files in "estonian" folder
					// kept in case someone wrongly configured language code / uses old translation directories
					$lang_formatted = "et";
					break;
				default:
					if (in_array(substr($language,0,2), $this->allowed_locales)) {
						$lang_formatted = substr($language,0,2);
					}
					break;
			endswitch;
		}
		return $lang_formatted;
	}
}
